///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//  Copyright  NetworkDLS 2002, All rights reserved
//
// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF 
// ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO 
// THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A 
// PARTICULAR PURPOSE.
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#ifndef _NSWFL_STRING_CPP_
#define _NSWFL_STRING_CPP_
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#include <Windows.H>
#include <StdIO.H>
#include <StdLib.H>

#include "NSWFL.H"

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//The returned pointer must be freed externally.
char *Space(int iSpaces)
{
	char *sSpaces = (char *) malloc(iSpaces + 1);
	if(sSpaces)
	{
		memset(sSpaces, ' ', iSpaces);
		sSpaces[iSpaces] = '\0';
		return sSpaces;
	}
	return NULL;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

char *Space(char *sBuffer, int iSpaces)
{
	memset(sBuffer, ' ', iSpaces);
	sBuffer[iSpaces] = '\0';
	return sBuffer;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool IsUpper(char cIn)
{
	return (cIn >= 65 && cIn <= 90);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool IsLower(char cIn)
{
	return (cIn >= 97 && cIn <= 122);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

char ToUpper(char cIn)
{
	if(IsLower(cIn))
	{
		return (cIn-32);
	}
	return cIn;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

char ToLower(char cIn)
{
	if(IsUpper(cIn))
	{
		return (cIn+32);
	}
	return cIn;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int IndexOf(const char *sBuf, const char cCharacter)
{
	return IndexOf(sBuf, (int)strlen(sBuf), cCharacter);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int IndexOf(const char *sBuf, int iBufSz, const char cCharacter)
{
	for(int iPos = 0; iPos < iBufSz; iPos++)
	{
		if((unsigned char)sBuf[iPos] == (unsigned char)cCharacter)
		{
			return iPos;
		}
	}
	return -1;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int LastIndexOf(const char *sBuf, const char cCharacter)
{
	return LastIndexOf(sBuf, (int)strlen(sBuf), cCharacter);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int LastIndexOf(const char *sBuf, int iBufSz, const char cCharacter)
{
	if(iBufSz > 0)
	{
		for(int iPos = iBufSz - 1; iPos > -1; iPos--)
		{
			if((unsigned char)sBuf[iPos] == (unsigned char)cCharacter)
			{
				return iPos;
			}
		}
	}
	return -1;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

char *UCase(char *sBuf, int iBufSz)
{
	for(int iPos = 0; iPos < iBufSz; iPos++)
	{
		sBuf[iPos] = ToUpper(sBuf[iPos]);
	}
	return sBuf;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

char *LCase(char *sBuf, int iBufSz)
{
	for(int iPos = 0; iPos < iBufSz; iPos++)
	{
		sBuf[iPos] = ToLower(sBuf[iPos]);
	}
	return sBuf;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

char *UCase(char *sBuf)
{
	return UCase(sBuf, (int)strlen(sBuf));
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

char *LCase(char *sBuf)
{
	return LCase(sBuf, (int)strlen(sBuf));
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int Trim(char *sInOutBuf, int iLen)
{
	int iLength = LTrim(sInOutBuf, iLen);
	return RTrim(sInOutBuf, iLen);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int Trim(char *sInOutBuf)
{
	return Trim(sInOutBuf, (int)strlen(sInOutBuf));
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int LTrim(char *sInOutBuf, int iLen)
{
	if(iLen > 0)
	{
		ReverseString(sInOutBuf, iLen);
		iLen = RTrim(sInOutBuf, iLen);
		ReverseString(sInOutBuf, iLen);
	}

	return iLen;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int LTrim(char *sInOutBuf)
{
	return LTrim(sInOutBuf, (int)strlen(sInOutBuf));
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int RTrim(char *sInOutBuf, int iLen)
{
    if(iLen > 0)
	{
		iLen = (GetLastCharPos(sInOutBuf, iLen) + 1);
		sInOutBuf[iLen] = '\0'; //replace the NULL terminator.
	}
	return iLen;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int RTrim(char *sInOutBuf)
{
	return RTrim(sInOutBuf, (int)strlen(sInOutBuf));
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

//sInOutBuf returns the string passed into it with all white spaces
//	trimmed off the beginning and end of it.
int CleanWhiteSpaces(char *sInOutBuf, int iLen)
{
	if(iLen > 0)
	{
		iLen = RTrim(sInOutBuf, iLen);
		ReverseString(sInOutBuf, iLen);
		iLen = RTrim(sInOutBuf, iLen);
		ReverseString(sInOutBuf, iLen);
	}
	return iLen;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int CleanWhiteSpaces(char *sInOutBuf)
{
	return CleanWhiteSpaces(sInOutBuf, (int)strlen(sInOutBuf));
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int ReplaceStringsI(const char *sInBuf, const char *sReplace, const char *sWith, char *sOutBuf, int iMaxSz)
{
	return ReplaceStringsI(sInBuf, sReplace, (int)strlen(sReplace), sWith, sOutBuf, iMaxSz);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int ReplaceStringsI(const char *sInBuf, const char *sReplace, int iReplaceLen,
				   const char *sWith, char *sOutBuf, int iMaxSz)
{
	const char *sNULL = NULL;

	int iReplaces = 0;

	if(sInBuf == NULL || sOutBuf == NULL)
	{
        return -1;
	}

    strcpy_s(sOutBuf, iMaxSz, "");

	for(sNULL = sInBuf; *sNULL; ++sNULL)
	{
		if( _strnicmp(sNULL, sReplace, iReplaceLen ) == 0 )
		{
			strcat_s(sOutBuf, iMaxSz, sWith);
			sNULL += (iReplaceLen - 1);
			iReplaces++;
		}
		else strncat_s(sOutBuf, iMaxSz, sNULL, 1);
	}

	return iReplaces;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int ReplaceStrings(const char *sInBuf, const char *sReplace, const char *sWith, char *sOutBuf, int iMaxSz)
{
	return ReplaceStrings(sInBuf, sReplace, (int)strlen(sReplace), sWith, sOutBuf, iMaxSz);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int ReplaceStrings(const char *sInBuf, const char *sReplace, int iReplaceLen,
				   const char *sWith, char *sOutBuf, int iMaxSz)
{
	const char *sNULL = NULL;

	int iReplaces = 0;

	if(sInBuf == NULL || sOutBuf == NULL)
	{
        return -1;
	}

    strcpy_s(sOutBuf, iMaxSz, "");

	for(sNULL = sInBuf; *sNULL; ++sNULL)
	{
		if( strncmp(sNULL, sReplace, iReplaceLen ) == 0 )
		{
			strcat_s(sOutBuf, iMaxSz, sWith);
			sNULL += (iReplaceLen - 1);
			iReplaces++;
		}
		else strncat_s(sOutBuf, iMaxSz, sNULL, 1);
	}

	return iReplaces;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int ReplaceCharacter(const char *sInBuf, char *sOutBuf, const char cReplace, const char cWith)
{
	int iPos = 0;
	int iOccur = 0;
	int iLength = (int)strlen(sInBuf);

	while(iPos < iLength)
	{
		if(sInBuf[iPos] != cReplace)
		{
			sOutBuf[iPos] = sInBuf[iPos];
			iOccur++;
		}
		else sOutBuf[iPos] = cWith;

		iPos++;
	}

	return iOccur;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int ReplaceCharacter(char *sInOutBuf, const char cReplace, const char cWith)
{
	return ReplaceCharacter(sInOutBuf, sInOutBuf, cReplace, cWith);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool ReverseString(char *sBuf, int iBufSz)
{
	char *String1 = NULL;
	char *String2 = NULL;

	if(!sBuf || !*sBuf)
	{
		return false;
	}

	for(String1 = sBuf, String2 = sBuf + iBufSz - 1; String2 > String1; ++String1, --String2)
	{
		*String1 ^= *String2;
		*String2 ^= *String1;
		*String1 ^= *String2;
	}

	return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int InStrI(const char *sSearchFor, int sSearchForSz, const char *sInBuf, const int iBufSz, const int iStartPos)
{
	int iLookingLoop  = 0; 

	if(iStartPos > iBufSz)
	{
		return -2;
	}
	if(sSearchForSz > iBufSz)
	{
		return -2;
	}

	for(int iControlLoop = iStartPos; iControlLoop <= (iBufSz-sSearchForSz); iControlLoop++)
	{
		if(ToLower(sInBuf[iControlLoop]) == ToLower(sSearchFor[iLookingLoop]))
		{
			while(iLookingLoop <= sSearchForSz) 
			{
				if(ToLower(sSearchFor[iLookingLoop]) == ToLower(sInBuf[iLookingLoop + iControlLoop]))
				{
					iLookingLoop++;
				}
				else iLookingLoop = sSearchForSz + 1; 

				if(iLookingLoop == sSearchForSz)
				{
					return iControlLoop;
				}
			}
		}
		iLookingLoop = 0;
	}

	return -1; 
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int InStr(const char *sSearchFor, int sSearchForSz, const char *sInBuf, const int iBufSz, const int iStartPos)
{
	int iLookingLoop  = 0; 

	if(iStartPos > iBufSz)
	{
		return -2;
	}
	if(sSearchForSz > iBufSz)
	{
		return -2;
	}

	for(int iControlLoop = iStartPos; iControlLoop <= (iBufSz-sSearchForSz); iControlLoop++)
	{
		if(sInBuf[iControlLoop] == sSearchFor[iLookingLoop])
		{
			while(iLookingLoop <= sSearchForSz) 
			{
				if(sSearchFor[iLookingLoop] == sInBuf[iLookingLoop + iControlLoop])
				{
					iLookingLoop++;
				}
				else iLookingLoop = sSearchForSz + 1; 

				if(iLookingLoop == sSearchForSz)
				{
					return iControlLoop;
				}
			}
		}
		iLookingLoop = 0;
	}

	return -1; 
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int InStr(const char *sSearchFor, const char *sInBuf, const int iBufSz, const int iStartPos)
{
	return InStr(sSearchFor, (int)strlen(sSearchFor), sInBuf, iBufSz, iStartPos);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int InStr(const char *sSearchFor, const char *sInBuf)
{
	return InStr(sSearchFor, sInBuf, (int)strlen(sInBuf), 0);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int InStrI(const char *sSearchFor, const char *sInBuf, const int iBufSz, const int iStartPos)
{
	return InStrI(sSearchFor, (int)strlen(sSearchFor), sInBuf, iBufSz, iStartPos);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int InStrI(const char *sSearchFor, const char *sInBuf)
{
	return InStrI(sSearchFor, sInBuf, (int)strlen(sInBuf), 0);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool CompareBinString(const char *sBuf1, const int iStrSz1, const char *sBuf2, const int iStrSz2)
{
	if(iStrSz1 != iStrSz2)
	{
        return false;
	}

	for(int iPos = 0; iPos < iStrSz1; iPos++)
	{
		if(((unsigned char)sBuf1[iPos]) != ((unsigned char)sBuf2[iPos]))
		{
            return false;
		}
	}

	return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool CompareNCString(const char *sBuf1, const char *sBuf2)
{
	int iStrSz1 = (int)strlen(sBuf1);
	int iStrSz2 = (int)strlen(sBuf2);

	if(iStrSz1 != iStrSz2)
	{
        return false;
	}

	for(int iPos = 0; iPos < iStrSz1 || iPos < iStrSz2; iPos++)
	{
		if(ToLower(sBuf1[iPos]) != ToLower(sBuf2[iPos]))
		{
            return false;
		}
	}

	return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool CompareCSString(const char *sBuf1, const char *sBuf2)
{
	int iStrSz1 = (int)strlen(sBuf1);
	int iStrSz2 = (int)strlen(sBuf2);

	if(iStrSz1 != iStrSz2)
	{
        return false;
	}

	for(int iPos = 0; iPos < iStrSz1 || iPos < iStrSz2; iPos++)
	{
		if(sBuf1[iPos] != sBuf2[iPos])
		{
            return false;
		}
	}

	return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int StringScan(const char *sInBuf, int iStartPos, int iLength, char *sOutVal, int iMaxSz)
{
	strncpy_s(sOutVal, iMaxSz, &sInBuf[iStartPos], iLength);
	return iLength;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int AppendStrings(const char *sBuf1, const char *sBuf2, int iBuf1Sz, int iBuf2Sz, char *sOutBuf, int iMaxSz)
{
	int iWPos = iBuf1Sz;
	int iRPos = 0;

	if(((iBuf1Sz + iBuf2Sz) + 1) > iMaxSz)
	{
		return -1;
	}

	memcpy_s(sOutBuf, iMaxSz, sBuf1, iBuf1Sz);

	while(iRPos < iBuf2Sz)
	{
		sOutBuf[iWPos++] = sBuf2[iRPos++];
	}

	sOutBuf[iWPos] = '\0';

	return iWPos;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int CompareFlagedString(const char *sBuf, const char *sFlag)
{
	int iFlagLength = (int)strlen(sFlag);

	if(strncmp(sBuf, sFlag, iFlagLength) == 0)
	{
		return iFlagLength;
	}

	return 0;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int GetFlagedValue(const char *sBuf, int iSkpLen, int iBufSz, char *sOutBuf, int iMaxSz)
{
	int iWPos = 0;
	int iRPos = iSkpLen;

	if(((iBufSz - iSkpLen) + 1) > iMaxSz)
	{
		return (0 - (iBufSz - iSkpLen)); // Returns the number of bytes required as a negitive number.
	}

	while(iRPos < iBufSz)
	{
		sOutBuf[iWPos++] = sBuf[iRPos++];
	}

	sOutBuf[iWPos] = '\0';

	return iWPos; //The length of the string returned in sOutBuf.
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

char *SimpleCipherString(char *sBuf, int iBufSz)
{
	for(int iPos = 0; iPos < iBufSz; iPos++)
	{
		((unsigned char *)sBuf)[iPos] += (unsigned char)128;
	}

	return sBuf;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int CopyString(char *sOutBuf, int iMaxSz, const char *sInBuf, int iInBufSz)
{
	int iPos = 0;

	if((iInBufSz + 1) > iMaxSz)
	{
		return -1;
	}

	while(iPos < iInBufSz)
    {
        sOutBuf[iPos] = sInBuf[iPos];
		iPos++;
    }

    sOutBuf[iPos] = '\0';

    return iPos;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//Returns the count of the Occurrences of "cLookFor" that are found in "sInStr".

int CountOccurrences(const char cLookFor, const char *sInStr, int iInStrSz)
{
	int iOccurrences = 0;
	for(int iPos = 0; iPos < iInStrSz; iPos++)
	{
		if((unsigned char)sInStr[iPos] == (unsigned char)cLookFor)
		{
			iOccurrences++;
		}
	}

	return iOccurrences;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//Returns the count of the Occurrences of "cLookFor" that are found in "sInStr".

int CountOccurrences(const char cLookFor, const char *sInStr)
{
	return CountOccurrences(cLookFor, sInStr, (int)strlen(sInStr));
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//Returns the count of the Occurrences of "sLookFor" that are found in "sInStr".

int CountOccurrences(const char *sLookFor, const char *sInStr, int iInStrSz)
{
	int iPos = 0;
	int iOccurrences = 0;
	while( (iPos = InStr(sLookFor, sInStr, iInStrSz, iPos)) >= 0)
	{
		iPos++;
		iOccurrences++;
	}

	return iOccurrences;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//Returns the count of the Occurrences of "sLookFor" that are found in "sInStr".

int CountOccurrences(const char *sLookFor, const char *sInStr)
{
	return CountOccurrences(sLookFor, sInStr, (int)strlen(sInStr));
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	Returns a zero-based, one-dimensional array containing a specified number of substrings.

	Example:

		char **sArray = NULL;
		char *sBuf = "Hello\tCruel\tWorld\tFrom\tC\tPlus\tPlus\t";
		int iItems = 0;
		int iItem = 0;

		sArray = SplitEx(sBuf, strlen(sBuf), sArray, &iItems, strlen("\t"), 1);

		printf("Array Items: %d\n", UBound(sArray));

		while(iItem < iItems)
		{
			printf("Item[%d]: %s\n", iItem, sArray[iItem]);
			iItem++;
		}

		FreeSplitStringEx(sArray, iItems);
*/

char **Split(const char *sBuf, int iBufSz, char **sArray, int *iOutItems, const char *sSplitOn, int iSplitOnSz)
{
	int iStartPos = 0;

	int iRPos = 0;
	int iWPos = 0;

	int iItems = 0;

	int iItemCount = 0; // The number of sub-strings that are going to be parsed from the string.
	int iLastPos = 0;

	//Get the number of items that we are going to split.
	while((iStartPos = InStr(sSplitOn, iSplitOnSz, sBuf, iBufSz, iStartPos)) >= 0)
	{
		iItemCount++;
		iStartPos = (iStartPos + iSplitOnSz);
		iLastPos = iStartPos;
	}

	//The string is not terminated by a delimiter.
	if(iBufSz > iLastPos)
	{
		iItemCount++;
	}

	iStartPos = 0;

	//Allocate enough RAM to hold all the sub-strings and the last NULL array item.
	sArray = (char **) calloc(sizeof(char *), iItemCount + 1);	

	while((iStartPos = InStr(sSplitOn, iSplitOnSz, sBuf, iBufSz, iRPos)) >= 0)
	{
		//Allocate enough RAM to hold the sub-string.
		sArray[iItems] = (char *) calloc(sizeof(char), (iStartPos - iRPos) + 1);

		iWPos = 0;
		while(iRPos < iStartPos)
		{
			sArray[iItems][iWPos++] = sBuf[iRPos++];
		}

		sArray[iItems][iWPos] = '\0';

		iRPos = (iRPos + iSplitOnSz);

		iItems++;
	}

	//The string is not terminated by a delimiter.
	if(iBufSz > iRPos)
	{
		//Allocate enough RAM to hold the sub-string.
		sArray[iItems] = (char *) calloc(sizeof(char), (iBufSz - iRPos) + 1);

		iWPos = 0;

		while(iRPos < iBufSz)
		{
			sArray[iItems][iWPos++] = sBuf[iRPos++];
		}

		sArray[iItems][iWPos] = '\0';
		iRPos = (iRPos + iSplitOnSz);

		iItems++;
	}
	
	//Insert a null at the end of the array.
	sArray[iItems] = NULL;

	*iOutItems = iItems;

	return sArray;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

char **Split(const char *sBuf, int iBufSz, char **sArray, int *iOutItems, const char *sSplitOn)
{
	return Split(sBuf, iBufSz, sArray, iOutItems, sSplitOn, (int)strlen(sSplitOn));
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	Returns a zero-based, one-dimensional array containing a specified number of substrings.

	Example:

		char **sArray = NULL;
		char *sBuf = "Hello\tCruel\tWorld\tFrom\tC\tPlus\tPlus\t";
		int iItem = 0;

		sArray = Split(sBuf, "\t");

		printf("Array Items: %d\n", UBound(sArray));

		while(sArray[iItem] != NULL)
		{
			printf("Item[%d]: %s\n", iItem, sArray[iItem]);
			iItem++;
		}

		FreeSplitString(sArray);
*/

char **Split(const char *sBuf, const char *sSplitOn)
{
	int iOutItems = 0;

	char **sArray = NULL;

	return Split(sBuf, (int)strlen(sBuf), sArray, &iOutItems, sSplitOn, (int)strlen(sSplitOn));
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

char **Split(const char *sBuf, int iBufSz, const char *sSplitOn)
{
	int iOutItems = 0;

	char **sArray = NULL;

	return Split(sBuf, iBufSz, sArray, &iOutItems, sSplitOn, (int)strlen(sSplitOn));
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function will free all the RAM allocated by an array.

	All array items, and the array its self.

	Intended for use with the Split() and SplitEx functions.
*/
void FreeArray(char **sArray, int iItems)
{
	if(sArray)
	{
		for(int iItem = 0; iItem < iItems; iItem++)
		{
			free(sArray[iItem]);
		}
		free(sArray);
	}
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function will free all the RAM allocated by an array.

	All array items, and the array its self.

	Intended for use with the Split() and SplitEx functions.
*/
void FreeArray(char **sArray)
{
	FreeArray(sArray, UBound(sArray));
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function will return the number of items in a char *Array[].
	This function will malfunction if the last item in the array is not NULL.

	Intended for use with the Split() and SplitEx functions.
*/
int UBound(char **sArray)
{
	int iItem = 0;

	while(sArray[iItem])
	{
		iItem++;
	}

	return iItem;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//Returns true if the passed character is 0-9.
bool IsNumeric(const char cIn)
{
	return (cIn >= 48 && cIn <= 57);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//Returns true if the passed string is in the following formats:
	//0, -0.0, 0.0, -0, -0.0
bool IsNumeric(const char *sText, int iLength)
{
	int iRPos = 0;
	bool bHasDecimal = false;

	if(iLength == 0)
	{
		return false;
	}

	if(sText[iRPos] == '-')
	{
		iRPos++;
	}

	for(; iRPos < iLength; iRPos++)
	{
		if(!IsNumeric(sText[iRPos]))
		{
			if(sText[iRPos] == '.')
			{
				if(iRPos == iLength-1) //Decimal cannot be the last character.
				{
					return false;
				}
				if(iRPos == 0 || (iRPos == 1 && sText[0] == '-')) //Decimal cannot be the first character.
				{
					return false;
				}


				if(bHasDecimal)
				{
					return false;
				}
				bHasDecimal = true;
			}
			else{
				return false;
			}
		}
	}

	return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool IsNumeric(const char *sText)
{
	return IsNumeric(sText, (int)strlen(sText));
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int Mid(const char *sInBuf, int iInBufSz, int iStartPos, int iRChars, char *sOutBuf, int iMaxSz)
{
	int iRPos = iStartPos;
	int iWPos = 0;
	int iChars = 0;

	if(iRChars <= 0 || iRChars > (iInBufSz - iStartPos) )
	{
		iChars = iInBufSz;
	}
	else iChars = (iStartPos + iRChars);

	if(((iChars - iStartPos) + 1) > iMaxSz)
	{
		return -1;
	}

	while(iRPos < iChars)
	{
		sOutBuf[iWPos++] = sInBuf[iRPos++];
	}

	sOutBuf[iWPos] = '\0';

	return iWPos;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int StrCat(char *sDest, int iMaxSz, int iDestLen, const char *sAppend, int iAppendLen)
{
	int iRPos = 0;
	int iWPos = iDestLen;

	if((iAppendLen + iDestLen) > iMaxSz)
	{
		return -1;
	}

	while(iRPos < iAppendLen)
	{
		sDest[iWPos++] = sAppend[iRPos++];
	}

	return iWPos;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int Right(char *sOutBuf, int iMaxSz, const char *sInBuf, int iInLen, int iRight)
{
	int iLen = iInLen;
	int iRPos = iLen - iRight;
	int iWPos = 0;

	if((iRight + 1) > iMaxSz)
	{
		return -1;
	}

	if(iRight > iLen)
	{
		strcpy_s(sOutBuf, iMaxSz, "");
		return 0;
	}

	while(iRPos < iLen)
	{
		sOutBuf[iWPos++] = sInBuf[iRPos++];
	}

	sOutBuf[iWPos] = '\0';

	return iWPos;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int Left(char *sOutBuf, int iMaxSz, const char *sInBuf, int iInLen, int iLeft)
{
	int iLen = iInLen;
	int iRPos = 0;

	if((iLeft + 1) > iMaxSz)
	{
		return -1;
	}

	if(iLeft > iInLen)
	{
		strcpy_s(sOutBuf, iMaxSz, "");
		return 0;
	}

	while(iRPos < iLeft)
	{
		sOutBuf[iRPos] = sInBuf[iRPos];
		iRPos++;
	}

	sOutBuf[iRPos] = '\0';

	return iRPos;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

//Returns true if the character passed into cChar is a whitespace, otherwise it returns false.
bool IsWhiteSpace(char cChar)
{
    if(cChar == ' ' || cChar == '\r' || cChar == '\n' || cChar == '\t' || cChar == '\0')
	{
        return true;
	}
    return false;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

//Returns the position of the first valid character in the string passed into sInBuf.
//iInBufSz should be the length of the string being passed into sInBuf
int GetFirstCharPos(const char *sInBuf, int iInBufSz)
{
    for(int iRPos = 0; iRPos < iInBufSz; iRPos++)
    {
        if(!IsWhiteSpace(sInBuf[iRPos]))
		{
            return iRPos;
		}
	}
    return -1;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int GetFirstCharPos(const char *sInBuf)
{
	return GetFirstCharPos(sInBuf, (int)strlen(sInBuf));
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

//Returns the position of the last valid character in the string passed into sInBuf.
//iInBufSz should be the length of the string being passed into sInBuf
int GetLastCharPos(const char *sInBuf, int iInBufSz)
{
    int iRPos = iInBufSz - 1;
    while(iRPos != -1)
    {
        if(!IsWhiteSpace(sInBuf[iRPos]))
		{
            return iRPos;
		}
		iRPos--;
	}
    return -1;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int GetLastCharPos(const char *sInBuf)
{
	return GetLastCharPos(sInBuf, (int)strlen(sInBuf));
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

char *FormatDouble(char *sOut, int iMaxSz, double dValue, int iDecimalPlaces)
{
	char sFormat[64];
	char sTemp[64];
	sprintf_s(sFormat, sizeof(sFormat), "%%.%df", iDecimalPlaces);
	sprintf_s(sTemp, sizeof(sTemp), sFormat, dValue);
	if(iDecimalPlaces > 0)
	{
		return FormatFloatString(sOut, iMaxSz, sTemp);
	}
	else {
		return FormatiString(sOut, iMaxSz, sTemp);
	}
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

char *FormatFloat(char *sOut, int iMaxSz, float fValue, int iDecimalPlaces)
{
	return FormatDouble(sOut, iMaxSz, fValue, iDecimalPlaces);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

char *FormatFloatString(char *sOut, int iMaxSz, const char *sIn)
{
	return FormatDoubleString(sOut, iMaxSz, sIn);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

char *FormatDoubleString(char *sOut, int iMaxSz, const char *sIn)
{
    int iRPos = (int)strlen(sIn) - 1;
    int iWPos = 0;
    int iDPos = 0;

	bool bFoundDecimal = false;

    if(iRPos < 3)
    {
        strcpy_s(sOut, iMaxSz, sIn);
        return sOut;
    }

    while(iRPos != -1)
    {
		if(sIn[iRPos] == '.')
		{
			bFoundDecimal = true;
			iDPos = -1;
		}

        if(iDPos++ == 3 && bFoundDecimal && sIn[iRPos] != '-')
        {
			Assert(iWPos >= iMaxSz, "Buffer too small.");
            sOut[iWPos++] = ',';
            iDPos = 1;
        }

		if(iWPos >= iMaxSz)
		{
			Assert(iWPos >= iMaxSz, "Buffer too small.");
		}
        sOut[iWPos++] = sIn[iRPos--];
    }

    ReverseString(sOut, iWPos);

    sOut[iWPos] = '\0';

    return sOut;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

char *FormatiString(char *sOut, int iMaxSz, const char *sIn)
{
    int iRPos = (int)strlen(sIn) - 1;
    int iWPos = 0;
    int iDPos = 0;

    if(iRPos < 3)
    {
        strcpy_s(sOut, iMaxSz, sIn);
        return sOut;
    }

    while(iRPos != -1)
    {
        if(iDPos++ == 3 && sIn[iRPos] != '-')
        {
			Assert(iWPos >= iMaxSz, "Buffer too small.");
            sOut[iWPos++] = ',';
            iDPos = 1;
        }

		Assert(iWPos >= iMaxSz, "Buffer too small.");
        sOut[iWPos++] = sIn[iRPos--];
    }

    ReverseString(sOut, iWPos);

    sOut[iWPos] = '\0';

    return sOut;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

char *FormatInteger(char *sOut, int iMaxSz, const long lIn)
{
    char sLong[64];
    _ultoa_s(lIn, sLong, sizeof(sLong), 10);
    return FormatiString(sOut, iMaxSz, sLong);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

char *I64FormatInteger(char *sOut, int iMaxSz, __int64 i64)
{
    char sLong[128];
    _i64toa_s(i64, sLong, sizeof(sLong), 10);
    return FormatiString(sOut, iMaxSz, sLong);
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

float fFormattedValue(const char *sFormatted)
{
	return (float) dFormattedValue(sFormatted);
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

double dFormattedValue(const char *sFormatted)
{
    int iLength = (int)strlen(sFormatted);
    int iRPos = 0;
    int iWPos = 0;

    char sVal[32];

	while(iRPos < iLength)
    {
        if(sFormatted[iRPos] == ',')
            iRPos++;

        if((sFormatted[iRPos] < 48 || sFormatted[iRPos] > 57) && sFormatted[iRPos] != '.')
            break;

        sVal[iWPos] = sFormatted[iRPos];
        iRPos++;
        iWPos++;
    }
    sVal[iWPos] = '\0';

    return atof(sVal);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

__int64 i64FormattedValue(const char *sFormatted)
{
    int iLength = (int)strlen(sFormatted);
    int iRPos = 0;
    int iWPos = 0;

    char sVal[32];

    if(iLength <= 3)
    {
        return atol(sFormatted);
    }

	while(iRPos < iLength)
    {
        if(sFormatted[iRPos] == ',')
            iRPos++;

        if(sFormatted[iRPos] < 48 || sFormatted[iRPos] > 57)
            break;

        sVal[iWPos] = sFormatted[iRPos];
        iRPos++;
        iWPos++;
    }
    sVal[iWPos] = '\0';

    return _atoi64(sVal);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int iFormattedValue(const char *sFormatted)
{
    int iLength = (int)strlen(sFormatted);
    int iRPos = 0;
    int iWPos = 0;
	bool bIsNegative = false;

    char sVal[32];

    if(iLength <= 3)
    {
        return atol(sFormatted);
    }

    if(sFormatted[0] == '-')
	{
		bIsNegative = true;
		iRPos++;
	}

	while(iRPos < iLength)
    {
        if(sFormatted[iRPos] == ',')
            iRPos++;

        if(sFormatted[iRPos] < 48 || sFormatted[iRPos] > 57)
            break;

        sVal[iWPos] = sFormatted[iRPos];
        iRPos++;
        iWPos++;
    }
    sVal[iWPos] = '\0';

	if(bIsNegative)
	{
		return atol(sVal) * -1;
	}
	else{
		return atol(sVal);
	}
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

unsigned int uiFormattedValue(const char *sFormatted)
{
	return (unsigned int)iFormattedValue(sFormatted);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int CharIndex(const char *sBuf, const char cChar)
{
	return CharIndex(sBuf, (int)strlen(sBuf), cChar);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int CharIndex(const char *sBuf, int iBufSz, const char cChar)
{
	for(int iPos = 0; iPos < iBufSz; iPos++)
	{
		if(sBuf[iPos] == cChar)
		{
			return iPos;
		}
	}

	return -1;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int SkipWhiteSpaces(const char *sBuf, int iBufSz, int iRPos)
{
	while(IsWhiteSpace(sBuf[iRPos]) && iRPos < iBufSz)
	{
		iRPos++;
	}

	return iRPos;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int SkipWhiteSpaces(const char *sBuf, int iBufSz, int *iRPos)
{
	while(IsWhiteSpace(sBuf[((int)*iRPos)]) && *iRPos < iBufSz)
	{
		((int)*iRPos)++;
	}

	return *iRPos;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

__inline bool IsCharInArray(const unsigned char cTokChar, const unsigned char *sChars, unsigned int iChars)
{
	for(unsigned int iChar = 0; iChar < iChars; iChar++)
	{
		if(cTokChar == sChars[iChar])
		{
			return true;
		}
	}
	return false;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	Returns the size in characters of the token.
*/
int GetToken(const char *sBuf, int iBufSz, char *sOutToken, int iMaxTokSz,
			 const char *sInvalidChars, int iInvalidCount)
{
	return GetNextToken(sBuf, iBufSz, sOutToken,
		iMaxTokSz, (int)0, sInvalidChars, iInvalidCount);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	Returns the size in characters of the token.
*/
int GetToken(const char *sBuf, int iBufSz, char *sOutToken, int iMaxTokSz)
{
	return GetNextToken(sBuf, iBufSz, sOutToken, iMaxTokSz, (int)0);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	Returns the size in characters of the token.
*/
int GetNextToken(const char *sBuf, int iBufSz, char *sOutToken,
				 int iMaxTokSz, int *iInOutRPos,
				 const char *sInvalidChars, int iInvalidCount)
{
	int iRPos = *iInOutRPos;
	int iWPos = 0;

	SkipWhiteSpaces(sBuf, iBufSz, &iRPos);

	if(sOutToken)
	{
		while(!IsCharInArray(sBuf[iRPos], (unsigned char *)sInvalidChars, iInvalidCount)
			&& iRPos < iBufSz && iWPos < iMaxTokSz)
		{
			sOutToken[iWPos++] = sBuf[iRPos++];
		}
		sOutToken[iWPos] = '\0';
	}

	SkipWhiteSpaces(sBuf, iBufSz, &iRPos);

	*iInOutRPos = iRPos;

	return iWPos; 
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	Returns the size in characters of the token.
*/
int GetNextToken(const char *sBuf, int iBufSz, char *sOutToken,
				 int iMaxTokSz, int *iInOutRPos)
{
	int iRPos = *iInOutRPos;
	int iWPos = 0;

	SkipWhiteSpaces(sBuf, iBufSz, &iRPos);

	if(sOutToken)
	{
		while(!IsWhiteSpace(sBuf[iRPos]) && iRPos < iBufSz && iWPos < iMaxTokSz)
		{
			sOutToken[iWPos++] = sBuf[iRPos++];
		}
		sOutToken[iWPos] = '\0';
	}

	SkipWhiteSpaces(sBuf, iBufSz, &iRPos);

	*iInOutRPos = iRPos;

	return iWPos; 
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	Returns the size in characters of the token.
*/
int GetNextToken(const char *sBuf, int iBufSz, char *sOutToken,
				 int iMaxTokSz, int iStartPos,
				 const char *sInvalidChars, int iInvalidCount)
{
	int iInOutRPos = iStartPos;
	return GetNextToken(sBuf, iBufSz, sOutToken, iMaxTokSz, &iInOutRPos, sInvalidChars, iInvalidCount);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	Returns the size in characters of the token.
*/
int GetNextToken(const char *sBuf, int iBufSz, char *sOutToken, int iMaxTokSz, int iStartPos)
{
	int iInOutRPos = iStartPos;
	return GetNextToken(sBuf, iBufSz, sOutToken, iMaxTokSz, &iInOutRPos);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool Abbreviate(const char *sString, char *Abbr, int iMaxSz)
{
	int iLen = (int)strlen(sString);
	int iRPos = 0;
	int iWPos = 0;
	int iSpaces = 0;

	if(iLen > 0)
	{
		Abbr[iWPos++] = sString[iRPos];

		while(iRPos < iLen)
		{
			if(sString[iRPos] == ' ')
			{
				if(iLen > iRPos)
				{
					Abbr[iWPos++] = sString[iRPos + 1];
				}
				iSpaces++;
			}
				
			iRPos++;
		}
	}

	if(iSpaces == 0)
	{
		strcpy_s(Abbr, iMaxSz, sString);
		return true;
	}

	Abbr[iWPos] = '\0';

	return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool ParseBoolean(const int iValue)
{
	return(iValue != 0);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool ParseBoolean(const char *sText)
{
	if(_strcmpi(sText, "yes") == 0
		|| _strcmpi(sText, "true") == 0
		|| _strcmpi(sText, "on") == 0)
	{
		return true;
	}

	return(ParseBoolean(atol(sText)));
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool IsMathChar(const char cChar)
{
	if(cChar == '*' || cChar == '/' || cChar == '~' || cChar == '-'
		|| cChar == '+' || cChar == '>' || cChar == '<' || cChar == '!'
		|| cChar == '&' || cChar == '|' || cChar == '^' || cChar == '%')
	{
		return true;
	}

	return false;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool IsAlNum(const char cChar)
{
	return ((cChar >= 'A' && cChar <= 'Z') ||
		(cChar >= 'a' && cChar <= 'z') ||
		(cChar >= '0' && cChar <= '9'));
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool IsAlpha(const char cChar)
{
	return ((cChar >= 'A' && cChar <= 'Z') || (cChar >= 'a' && cChar <= 'z'));
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool IsHexDigit(const char cChar)
{
	return ((cChar >= 'A' && cChar <= 'F') ||
		(cChar >= 'a' && cChar <= 'f') ||
		(cChar >= '0' && cChar <= '9'));
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#endif
